% Test environment for fdpred.pl
% (c) Peter Szeredi, November 2002

:- use_module(library(clpfd)).
:- use_module(library(lists)).
:- use_module(fdpred).

tests :-
	check_fd_pred_negative,
	separator,
	check_fd_pred_positive,
	separator,
	check_indexicals_negative,
	separator,
	check_indexicals_positive.

separator :-
	format('~*c\n', [70,0'-]).

check_fd_pred_negative :-
	Pred = [('x=<y=<z?'(X, Y, Z) +:  
		              Y in min(X)..max(Z), 
		              Z in min(Y).. sup,   
		              X in inf..max(Y) )],
	check_fd_pred(Pred, 1, 3).

check_fd_pred_positive :-
	Pred = [(
		'x\\=y'(X,Y) +:    
		     X in \{Y},	   
		     Y in \{X}	   
		),(		   
		'x\\=y'(X,Y) -:    
		     X in dom(Y),  
		     Y in dom(X)   
		),(		   
		'x\\=y'(X,Y) +?	   
		     X in \dom(Y)  
		),(		   
		'x\\=y'(X,Y) -?	   
		     X in {Y}	   
		)],
	check_fd_pred(Pred, 1, 3).

check_indexicals_negative :-
	check_indexical('z>max(x,y)'(X,Y,Z),
			X in inf..max(Z)-1,
			1, 3,
			Z > max(X, Y)
		       ),
	check_indexical('x=<y=<z'(X, Y, Z),
			Z in min(Y)..sup, 
			1, 3,
			(X=<Y, Y=<Z)
		       ),
	check_indexical('z>max(x,y)'(X,Y,Z),
			X in {1},
			0, 2,
			Z > max(X, Y)
		       ).

check_indexicals_positive :-
	check_indexical('x+y=t tsz'(X, Y, T),
			X in dom(T) - dom(Y),
			0, 9,
			X+Y=:=T
		       ),
	check_indexical(no_threat_4(X, Y),
			X in (4..card(Y))?(inf..sup) \/
		              unionof(B,dom(Y),\{B,B+1,B-1}),
			0, 9,
			(X=\=Y,X=\=Y+1,X=\=Y-1)
		       ),
	check_indexical('x=<y=<z'(X, Y, Z),
			Z in ((inf..max(Y)) /\ dom(X)) ? (min(Y)..sup), 
			0, 9,
			(X=<Y, Y=<Z)
		       ),
	check_indexical(p(I, X, Y, Z),
			Z in unionof(J, dom(I) mod 2,
				     switch(J, [0-dom(X),1-dom(Y)])),
			0, 9,
			(I mod 2 =:= 0 -> Z = X; Z = Y)
		       ).

check_fd_pred(Pred, Min, Max) :-
	fd_pred_header(Pred, Min, Max),
	fd_pred_semantics(Pred, Min, Max, Set0, Diffs),
	format('Semantics of the first indexical:\n', []),
	show_semantics(Set0),
	(   Diffs = [] ->
	    format('All other indexicals have the same semantics\n', [])
	;   format('Indexicals with different semantics:\n', []),
	    member(ClNo/IndNo-Set, Diffs),
	    format('Semantics of indexical ~w in clause ~w:\n',
			 [IndNo,ClNo]),
	    show_semantics(Set),
	    fail
	; true
	).

fd_pred_header(Pred, Min, Max) :-
	numbervars(Pred, 23, _),
	format('Checking indexicals within predicate:\n', []),
	(   member(Cl, Pred),
	    Cl =.. [Neck,Head,Body],
	    format('~q ~q\n    ~q.\n', [Head,Neck,Body]),
	    fail
	;   true
	),
	format('using interval ~w..~w ...\n', [Min,Max]),
	fail.
fd_pred_header(_, _, _).


show_semantics(Rel) :-
	(   member(H, Rel),
	    format('      ~q.\n', [H]),
	    fail
	;   true
	).

check_indexical(Head, Ind, Min, Max, Rel) :-
	ind_header(Head, Ind, Min, Max, Rel),
	check_indexical(Head, Ind, Min, Max, Rel, Extra, Missing),
	(   Extra = [], Missing = [] ->
	    format('   ... No discrepancy found.\n', [])
	;   member(Head, Extra),
	    format('   ~q holds,         while ~q is not true\n', [Head,Rel]),
	    fail
	;   member(Head, Missing),
	    format('   ~q does not hold, while ~q is true\n', [Head,Rel]),
	    fail
	;   true
	).

ind_header(Head, Ind, Min, Max, Rel) :-
	numbervars(Head, 23, _),
	numbervars(Ind, 0, _),
	format('Comparing indexical ~q\n   with relation ~q, \n',
	       [Ind,Rel]),
	format('   using interval ~w..~w ...\n', [Min,Max]),
	fail.
ind_header(_, _, _, _, _).


check_indexical(Head, Ind, Min, Max, Rel, Extra, Missing) :-
	indexical_semantics((Head+:Ind), Min, Max, IndSet),
	relation_semantics((Head:-Rel), Min, Max, RelSet),
	findall(H, (member(H, IndSet), non_member(H, RelSet)), Extra),
	findall(H, (member(H, RelSet), non_member(H, IndSet)), Missing).
