% Calculating FD predicate semantics
% (c) Peter Szeredi, November 2002

:- module(fd_semantics, [fd_pred_semantics/5,
		       indexical_semantics/4,
		       relation_semantics/4]).

:- use_module(library(clpfd)).
:- use_module(library(lists)).
:- use_module(library(terms)).

% fd_pred_semantics(+Pred, +Min, +Max, -Set0, Diffs) :-

% Pred is an FD predicate, represented as a 1..4 long list, containg the
% clauses with different neck symbols.  The heads of these clauses should
% be the same (modulo variable renaming). Min..Max is the interval in which
% the relation defined by Pred is to be calculated.
% Output arguments: 
% Set0 is the relation represeneted by the very first indexical in the
% predicate, represented as a sorted list of head instances. Diffs lists
% those further indexicals whose semantics differs. Each element of Diffs is
% of form; ClNo/IndNo-Set, semantics: the indexical number IndNo within
% clause ClNo represents a relation Set (which is different from Set0).

% Note that the semantics of the indexicals in negative clauses is calculated
% as the complement of the range expression on its right hand side.

fd_pred_semantics(Pred, Min, Max, Set0, Diffs) :-
	findall(Sel-Ind,
		indexical_in_pred(Sel, Ind, Pred), [_-Ind0|SInds]),
	indexical_semantics(Ind0, Min, Max, Set0),
	findall(Sel-Set,
	        (   member(Sel-Ind,SInds),
		    indexical_semantics(Ind, Min, Max, Set),
		    Set \= Set0
		),
		Diffs).

% indexical_semantics(+HInd, +Min, +Max, -Set):
% Set is the semantics of HInd, calculated in interval Min..Max.
% HInd is of form (Head+:Ind), where Ind is a single indexical.
% Set is returned as a sorted list of Head instances.
indexical_semantics((Head+:Ind), Min, Max, Set) :-
        warn_indexical_vars(Head, Ind),
	all_solutions(Head, solve_indexical(Ind), Min, Max, Set).

% Issue a warning if the set of vars of Head and Ind are different.
warn_indexical_vars(Head, Ind) :-
	term_variables(Head, HVs),
	term_variables(Ind, IVs),
	(   HVs == IVs -> true
	;   numbervars(Head, 23, _),
	    numbervars(Ind, 0, _),
	    member(HV, HVs),
	    (   member(V, IVs), V == HV  -> true
	    ;   print_message(warning, head_var(HV, does_not_appear_in(Ind)))
	    ), fail
	;   numbervars(Head, 23, _),
	    numbervars(Ind, 0, _),
	    member(IV, IVs),
	    (   member(V, HVs), V == IV  -> true
	    ;   print_message(error, indexical_var(IV, does_not_appear_in_head(Head)))
	    ), fail
	;   true
	).

% relation_semantics(+HRel, +Min, +Max, -Set):
% Set is the semantics of HRel, calculated in interval Min..Max.
% HRel is of form (Head:-Rel), where Rel is an arbitrary Prolog goal.
% Set is returned as a sorted list of Head instances.
relation_semantics((Head:-Rel), Min, Max, Set) :-
	all_solutions(Head, Rel, Min, Max, Set).

% Auxiliary predicates

indexical_in_pred(Sel, (Head +: Ind), Clauses) :-
	nth(ClNo, Clauses, Clause),
	Clause =.. [Neck,Head,Body],
	round_list_to_list(Body, Indexicals, []),
	nth(IndNo, Indexicals, Ind0),
	(   positive_neck(Neck) -> Ind = Ind0
	;   Ind0 = (X in R), Ind = (X in \(R))
	),
	Sel = ClNo/IndNo.

positive_neck(+:).
positive_neck(+?).
	
solve_indexical(X in R) :-
	range_val(R, S),
	fdset_member(X, S).

all_solutions(Head, Goal, Min, Max, Sols) :-
	Head =.. [_|Vars],
	findall(Head, (	  domain(Vars, Min, Max),
			  labeling([], Vars),
			  Goal), Sols0),
	sort(Sols0, Sols).

range_val(dom(X), S) :-
	fdset_singleton(S, X).
range_val({Ts}, S) :-
	round_list_to_list(Ts, TL, []),
	terms_vals(TL, VL),
	list_to_fdset(VL, S).
range_val(T1..T2, S) :-
	term_val(T1, V1),
	term_val(T2, V2),
	(   fdset_interval(S, V1, V2) -> true
	;   empty_fdset(S)
	).
range_val(R1/\R2, S) :-
	range_val(R1, S1),
	range_val(R2, S2),
	fdset_intersection(S1, S2, S).
range_val(R1\/R2, S) :-
	range_val(R1, S1),
	range_val(R2, S2),
	fdset_union(S1, S2, S).
range_val(\R1, S) :-
	range_val(R1, S1),
	fdset_complement(S1, S).
range_val(R1+R2, S) :-
	range_val(R1, S1),
	range_or_term_val(R2, S2),
	pointwise_binop(+, S1, S2, S).
range_val(-R1, S) :-
	range_val(R1, S1),
	fdset_singleton(S0, 0),
	pointwise_binop(-, S0, S1, S).
range_val(R1-R2, S) :-
	(   range_val(R1, S1) -> 
	    range_or_term_val(R2, S2)
	;   term_val(R1, S1),
	    range_val(R2, S2)
	),
	pointwise_binop(-, S1, S2, S).
range_val(R1 mod R2, S) :-
	range_val(R1, S1),
	range_or_term_val(R2, S2),
	pointwise_binop(mod, S1, S2, S).
range_val(R1 ? R2, S) :-
	range_val(R1, S1),
	(   empty_fdset(S1) -> empty_fdset(S)
	;   range_val(R2, S)
	).
range_val(unionof(X,R1,R2), S) :-
	range_val(R1, S1),
	findall(S, (   fdset_member(X, S1), range_val(R2, S)   ), Ss),
	fdset_union(Ss, S).
range_val(switch(T1,MAPLIST), S) :-
	term_val(T1, V),
	(   member(V-R, MAPLIST) -> range_val(R, S)
	;   empty_fdset(S)
	).

range_or_term_val(T, S) :-
	term_val(T, V), !, fdset_singleton(S, V).
range_or_term_val(R, S) :-
	range_val(R, S).

pointwise_binop(Op, S1, S2, S) :-
	Exp =.. [Op,V1,V2],
	findall(V, (   fdset_member(V1, S1),
		       fdset_member(V2, S2),
		       V is Exp
		   ),
		Vs),
	list_to_fdset(Vs, S).

terms_vals([], []).
terms_vals([T|Ts], [V|Vs]) :-
	term_val(T, V),
	terms_vals(Ts, Vs).

term_val(Term, V) :-
	tval(Term, V0), !,
	(   V0 =:= +inf -> V = sup
	;   V0 =:= -inf -> V = inf
	;   integer(V0), V = V0
	).
term_val(Term, _) :-
	print_message(error, type_error(integer,Term)), fail.

tval(Exp, _) :- var(Exp), !, fail.
tval(min(X), X) :- integer(X).
tval(max(X), X) :- integer(X).
tval(card(X), 1) :- integer(X).
tval(I, I) :-
	integer(I).
tval(inf, -inf).
tval(sup, +inf).
tval(-T, V) :-
	tval(T, V0),
	V is -V0.
tval(Exp, V) :-
	Exp =.. [F,T1,T2],
	tval(T1, V1),
	tval(T2, V2),
	do_binop(F, V1, V2, V).

do_binop(Op, V1, V2, V) :-
	simple_binop(Op),
	VExp =.. [Op,V1,V2],
	V is VExp.
do_binop(Op, V1, V2, V) :-
	divlike_expr(Op, V1, V2, VExp),
	(   float(V2) -> fail % ie. V2= +inf or V2= -inf
	;   V2 == 0 -> fail
	;   float(V1) -> V = V1  
	;   V is VExp
	).

divlike_expr(mod, V1, V2, V1 mod V2).
divlike_expr(/<, V1, V2, integer(floor(V1/V2))).
divlike_expr(/>, V1, V2, integer(ceiling(V1/V2))).

simple_binop(+).
simple_binop(-).
simple_binop(*).

% utility

round_list_to_list((R1,R2)) --> !,
	round_list_to_list(R1),
	round_list_to_list(R2).
round_list_to_list(R) --> [R].
