% Environment for checking FD predicate semantics
% (c) Peter Szeredi, November 2002

% Load this file to have your FD predicates checked against a Prolog relation.
% Directives:

% 
% :- fd_pred_semantics(Head, Rel).
%
%    Include this in front of the FD predicate defining Head.
%    Rel is an arbitrary Prolog goal which tests the truth of Head.
%    Example:
%               :- fd_pred_semantics('z>max(x,y)'(X,Y,Z), Z>max(X,Y)).
%    (This is the default, to support KHF3.)

% :- fd_test_range(Min, Max).
%
%    The semantics of the FD-pred and the Prolog Rel will be compared
%    by evaluating both on all instantiations of head variables in
%    range Min..Max.
%    (The default is 1..2, as this is enough for KHF3.)

% :- fd_check_options(Opts).
%
%    Opts is a list of options. If this list contains the atom `verbose',
%    then fdcheck will produce output even if no discrepancy has been found.
%    The default is Opt = [].

:- module(fd_check, []).
:- use_module(library(clpfd)).
:- use_module(library(lists)).
:- use_module(fdpred).

fd_neck(+:, +).
fd_neck(+?, +).
fd_neck(-:, -).
fd_neck(-?, -).

fd_clause_check(FDClause) :-
	FDClause =.. [Neck,Head,Body],
	fd_neck(Neck, PosNeg),
	bb_get('$fdcheck':semantics,(Head:-Rel)),
	bb_get('$fdcheck':test_range, Min..Max),
	bb_get('$fdcheck':options, Opts),
	fd_semantics:round_list_to_list(Body, Inds, []),
	nth(N, Inds, Ind0),
	(   PosNeg = + -> Ind = Ind0
	;   Ind0 = (X in R), Ind = (X in \(R))
	),
	check_indexical(Head, Ind, Min, Max, Rel, Extra, Missing),
	(   Extra = [], Missing = [] ->
	    (   memberchk(verbose, Opts),
		ind_header(Neck, N, Head, Ind, Min, Max, Rel), 
		format('   ... No discrepancy found.\n', [])
	    ;   true
	    )
	;   ind_header(Neck, N, Head, Ind, Min, Max, Rel), fail
	;   member(Head, Extra),
	    format('   ~q holds,         while ~q is not true\n', [Head,Rel]),
	    fail
	;   member(Head, Missing),
	    format('   ~q does not hold, while ~q is true\n', [Head,Rel]),
	    fail
	;   memberchk(store, Opts) ->
	    retractall('$fderror':error(Head, Neck, N, _, _)),
	    assert('$fderror':error(Head, Neck, N, Extra, Missing))
	;   true
	).

ind_header(Neck, N, Head, Ind, Min, Max, Rel) :-
	numbervars(Head, 23, _),
	numbervars(Ind, 0, _),
	format('Comparing clause ~w , indexical No. ~w: ~q\n   with relation ~q, \n',
	       [Neck,N,Ind,Rel]),
	format('   using interval ~w..~w ...\n', [Min,Max]),
	fail.
ind_header(_, _, _, _, _, _, _).


check_indexical(Head, Ind, Min, Max, Rel, Extra, Missing) :-
	indexical_semantics((Head+:Ind), Min, Max, IndSet),
	relation_semantics((Head:-Rel), Min, Max, RelSet),
	findall(H, (member(H, IndSet), non_member(H, RelSet)), Extra),
	findall(H, (member(H, RelSet), non_member(H, IndSet)), Missing).


:- multifile user:term_expansion/2.
:- dynamic user:term_expansion/2.
user:term_expansion((:- fd_pred_semantics(Head, Rel)),
	       (:-bb_put('$fdcheck':semantics, (Head:-Mod:Rel)))
	       ) :-
	prolog_load_context(module, Mod).
user:term_expansion((:- fd_test_range(Min, Max)), 
	       (:-bb_put('$fdcheck':test_range, Min..Max))
	      ).
user:term_expansion((:- fd_check_options(Opts)),
	       (:-bb_put('$fdcheck':options, Opts))
	      ).

:- Head = user:term_expansion(FDClause, _),
   Body = (   fd_check:current_predicate(fd_check:fd_clause_check/1),
	      fd_check:fd_clause_check(FDClause),
	      fd_check:fail
	  ),
   (   clause(Head, Body) -> true
   ;   asserta((Head :- Body))
   ).

:- bb_put('$fdcheck':test_range, 1..2).
:- bb_put('$fdcheck':options, []). % [verbose]
:- bb_put('$fdcheck':semantics, ('z>max(x,y)'(X,Y,Z):-Z>max(X,Y))).
