/**
 * @author Laszlo Siroki
 */

includes ioboard;

includes byteorder;
includes crc;
includes IA4420;

module IA4420RadioM {
	provides {
		interface StdControl;
		interface BareSendMsg as Send;
		interface ReceiveMsg as Receive;

		interface IA4420Control;
		
	}
	uses {
		//interface IA4420Spi;

		interface HPLIA4420Command;
		interface IA4420Interrupt;

		interface Random;

#ifdef RADIO_DEBUG
//		interface IntOutput as Debug;
		interface DiagMsg;
		interface Timer;
#endif		

		interface TimerJiffyAsync;
	}
}

implementation {

enum {
	RX_DISABLED,
	RX_INIT,
	RX_ENABLING,
	RX_DISABLING_PATTERN_RECOG,
	RX_ENABLING_PATTERN_RECOG,
	RX_WAITING_FOR_FIRST_BYTE,
	RX_WAITING_FOR_NEXT_BYTE,
	RX_READING_FIFO,
	RX_DISABLING_PATTERN_RECOG2,
	RX_ERROR,
	RX_SUSPENDED
};
	uint8_t rxstate;

enum {
	TX_DISABLED,
	TX_NO_PACKET_PENDING,
	TX_PACKET_PENDING,
	TX_WAITING,
	TX_LISTENING,
	TX_DISABLING_RECEIVER,
	TX_WRITING_PREAMBLE1,
	TX_WRITING_PREAMBLE2,
	TX_STARTING_TRANSMISSION,
	TX_WRITING_PREAMBLE3,
	TX_WRITING_PREAMBLE4,
	TX_SENDING_SYNC1,
	TX_SENDING_SYNC2,
	TX_SENDING,
	TX_SENDING_DUMMY,
	TX_DISABLING_TRANSMITTER,
	TX_ENABLING_RECEIVER
};
	uint8_t txstate;


	TOS_MsgPtr txbufptr;  // pointer to transmit buffer
	TOS_MsgPtr rxbufptr;  // pointer to receive buffer
	TOS_Msg RxBuf;	// save received messages
	uint16_t crc;
	
	uint8_t *rxbyteptr;

	uint8_t *byteptr, *endbyteptr;
	
	bool waitforradio = FALSE;
/*
	bool packetreceivedpending = FALSE;

	bool receivestartpending = FALSE;
*/
	bool txpending = FALSE;
	bool txstartpending = FALSE;

	
enum {
	RX_ENABLED_BIT = 1,
	TX_ENABLED_BIT = 2,
	TX_WAIT_ENABLED_BIT = 4,
	TX_LISTEN_ENABLED_BIT = 8
};

	uint8_t mode = RX_ENABLED_BIT | TX_ENABLED_BIT | TX_WAIT_ENABLED_BIT | TX_LISTEN_ENABLED_BIT;

	// Who has the control over the radio hardware?
enum {
	NO_CONTROLLER,
	RX_CONTROLLED,
	TX_CONTROLLED,
	CON_CONTROLLED
};
	uint8_t radio_ctrl = NO_CONTROLLER;


/*
enum {
	COMM_IDLE = 0,
	COMM_SETFREQ,
	COMM_SETDATARATE,
	COMM_SETTXPOWER
};
*/

enum {
	FREQ_DIRTY = 1,
	DATARATE_DIRTY = 2,
	TXPOWER_DIRTY = 4
};
	uint8_t ctrlstate = 0;

	bool control_pending = FALSE;
	bool control_executing = FALSE;

	uint16_t newfreq;
	uint8_t newdatarate;
	uint8_t newtxpower;
	
	
		
	volatile uint8_t sync_done;


/*********************
*   Local functions  *
*********************/

	result_t getControl(uint8_t who) {
		result_t res;
		atomic {
//			if (radio_ctrl == who)
//				res=SUCCESS;
//			else
			if(radio_ctrl == NO_CONTROLLER) {
				radio_ctrl = who;
				res=SUCCESS;
			} else
				res = FAIL;
		}
		return res;
	}

	result_t releaseControl(uint8_t who) {
		result_t res;
		atomic {
			if (radio_ctrl == who) {
				radio_ctrl = NO_CONTROLLER;
				res=SUCCESS;
			} else 
				res = FAIL;
		}
		return res;
	}

	result_t hasControl(uint8_t who) {
		result_t res;
		atomic {
			if (radio_ctrl == who) {
				res=SUCCESS;
			} else 
				res = FAIL;
		}
		return res;
	}

	bool do_control_pending();
	bool do_tx_pending();
	
	bool do_pending_thing(){
		return (do_control_pending() || do_tx_pending());
	}

void fatalerror() {
			TOSH_MAKE_GPIO_15_OUTPUT();
			TOSH_SET_GPIO_15_PIN();

			TOSH_SET_LED_3_PIN();
			
}
/*
void wait() {
	uint32_t i;
	for(i=0;i<2000000;i++) ;
}
*/

void wait_for_spi() {
	uint8_t lsync_done = 0;

	while (lsync_done == 0)
		atomic {lsync_done = sync_done;}

	atomic {
		sync_done = 0;
	}

}

#ifdef RADIO_DEBUG

	task void valami() {
		if (call DiagMsg.record()) {
			call DiagMsg.str("hiba!"); 
			call DiagMsg.send();
		}
	}
#endif

/********************************/
/** init/start/stop functions  **/
/********************************/

	task void inittask();
	result_t rxMode();
	
	command result_t StdControl.init() {
		TOSH_MAKE_IA4420_NSEL_OUTPUT();	// nSEL pin
		TOSH_MAKE_IA4420_NIRQ_INPUT();  // nIRQ pin
		
		TOSH_MAKE_GPIO_14_INPUT(); // nIRQ Connected with this pin also in current implementation

		TOSH_MAKE_LED_1_OUTPUT();
		TOSH_MAKE_LED_2_OUTPUT();
		TOSH_MAKE_LED_3_OUTPUT();
		TOSH_MAKE_LED_4_OUTPUT();
		TOSH_CLR_LED_1_PIN();
		TOSH_CLR_LED_2_PIN();
		TOSH_CLR_LED_3_PIN();
		TOSH_CLR_LED_4_PIN();

		call Random.init();

		atomic {
			rxstate = RX_DISABLED;
			txstate = TX_DISABLED;
			txpending = FALSE;
			rxbufptr = &RxBuf;
		}
		call IA4420Interrupt.checkInterrupt();
		// Radio initialisation will be done by POR interrupt.
		return SUCCESS;
	}

	task void blinktask() {
		static uint8_t led=0;
		if (!led) {
			TOSH_SET_LED_4_PIN();
			led=1;
		} else {	
			TOSH_CLR_LED_4_PIN();
			led=0;
			if (!TOSH_READ_BUTTON_1_PIN()) {
				call IA4420Interrupt.checkInterrupt();
				TOSH_CLR_LED_3_PIN();
#ifdef RADIO_DEBUG
				call Timer.start(TIMER_ONE_SHOT, 1000);
#endif
			}
		}
		post blinktask();
	}
// Start and stop are useless now
	command result_t StdControl.start() {
		post blinktask();
		return SUCCESS;
	}
  
	command result_t StdControl.stop() {
		return SUCCESS;
	}

	void com_r04_s01__Init_433 (unsigned int sub_frequency, uint8_t baseband_bw, uint8_t deviation, uint8_t power, uint8_t data_rate)
	{
		uint8_t rssitreshold = 3;
		atomic {
			rxstate = RX_INIT;
			sync_done = 0;
		}

		// Configuration Setting command
		call HPLIA4420Command.CSC(0xD7);// enables the internal data register, rx fifo, 433 MHz, 12 pF
		wait_for_spi();

		// Configuration Setting command again experiences shows that it should be done twice
		call HPLIA4420Command.CSC(0xD7);// enables the internal data register, rx fifo, 433 MHz, 12 pF
		wait_for_spi();
	  
		// Sub frequencies
		call HPLIA4420Command.FSC(sub_frequency);
		wait_for_spi();

		// Power management command synthesizer, osc. on
		call HPLIA4420Command.PMC(0x18);
		wait_for_spi();
		
		// Setting baseband BW and RSSI treshold
		call HPLIA4420Command.RCC(0x0600 | baseband_bw | rssitreshold);
		wait_for_spi();

		// Transmiter Control command
		call HPLIA4420Command.TXCCC(deviation|power);
		wait_for_spi();

		// Data rate command
		call HPLIA4420Command.DRC(data_rate);
		wait_for_spi();
	}


// Initialisation task: currently non-split-phase
	task void inittask() {
		com_r04_s01__Init_433(
			IA4420_DEFAULT_FREQ,
			IA4420_DEFAULT_BASEBAND_BW,
			IA4420_DEFAULT_FSK_DEVIATION,
			IA4420_DEFAULT_OUTPUT_POWER,
			IA4420_DEFAULT_DATA_RATE
			);

		atomic {
			rxstate = RX_DISABLED;
			txstate = TX_DISABLED;
//			mode = ...;
		}
#ifdef RADIO_DEBUG
		if (call DiagMsg.record()) {
			call DiagMsg.str("Init done."); 
			call DiagMsg.uint16(TOS_LOCAL_ADDRESS); 
			call DiagMsg.send();
		}
#endif

		rxMode();

#ifdef RADIO_DEBUG
//		if (call DiagMsg.record()) {
//			call DiagMsg.str("rx on"); 
//			call DiagMsg.send();
//		}
#endif
		atomic txstate = TX_NO_PACKET_PENDING;

	//	call Timer.start(TIMER_REPEAT, 1000);
	}



/***********************
*   Receive functions  *
************************/

	void do_on_rx_error();
	void do_enable_rx();
	void do_disable_pattrecog();
	void do_enable_pattrecog();
	void do_wait_for_first_byte();
	void do_read_fifo();
	void do_wait_for_next_byte();
	void do_disable_pattrecog2();
	void do_finish_packet_receive();

	void do_wait_for_tx();	

/** Change to RX mode  **/

	result_t rxMode() {
		uint8_t rxs;
//		result_t ret = FAIL;
		atomic {
			rxs = rxstate;
 		}

		if (rxs == RX_DISABLED)
		{		
			do_enable_rx();
			return SUCCESS;
		}
		return FAIL;	

	}

/** Signal packet reception **/

	task void PacketRcvd() {
		TOS_MsgPtr pBuf;
		uint8_t lgt, local=0;
		
		atomic {
			pBuf = rxbufptr;
		}
		lgt = pBuf->length;
		pBuf = signal Receive.receive((TOS_MsgPtr)pBuf);

//		TOSH_CLR_LED_1_PIN();
		
	//	wait();		

		atomic {
			if (pBuf)
				rxbufptr = pBuf;
/*			packetreceivedpending = FALSE;
			
			if (receivestartpending)
			{
				receivestartpending = FALSE;
				local = 1;
			}
*/ 
		}
/*
		if (local == 1)
			do_enable_pattrecog();
*/
#ifdef RADIO_DEBUG
/*		
		if (call DiagMsg.record()) {
			call DiagMsg.str("Packet received!"); 
			call DiagMsg.int8(lgt); 
			call DiagMsg.int8(local); 
			call DiagMsg.int8(rxbufptr->data[0]); 
			call DiagMsg.send();
		}
*/ 
#endif

		do_disable_pattrecog();
	}


	void do_on_rx_error() {
		atomic {
			waitforradio = FALSE;
			rxstate = RX_ERROR;
		}
#ifdef RADIO_DEBUG

		if (call DiagMsg.record()) {
			call DiagMsg.str("ERR"); 
			call DiagMsg.send();
		}

#endif
		if (!call HPLIA4420Command.FRMC(0x80)) // disable pattern recog
			fatalerror();
	}

	void do_enable_rx() {
		if (getControl(RX_CONTROLLED)) {
			atomic {
				rxstate = RX_ENABLING;
			}
			call HPLIA4420Command.PMC(0xD8); // RX mode

#ifdef RADIO_DEBUG
			if (call DiagMsg.record()) {
				call DiagMsg.str("Enabling RX"); 
				call DiagMsg.send();
			}
#endif
		}	
	}

	void do_disable_pattrecog() {
		atomic {
			waitforradio = FALSE;
			rxstate = RX_DISABLING_PATTERN_RECOG;
		}

#ifdef RADIO_DEBUG

		if (call DiagMsg.record()) {
			call DiagMsg.str("DPR"); 
			call DiagMsg.send();
		}

#endif

		if (!call HPLIA4420Command.FRMC(0x80)) // disable pattern recog
			fatalerror();
	}

	void do_enable_pattrecog() {
		atomic {
			rxstate = RX_ENABLING_PATTERN_RECOG;
			rxbyteptr = (uint8_t *) rxbufptr;
			crc = 0;
		}

		//wait();
		
		call HPLIA4420Command.FRMC(0x82); // enable pattern recog


#ifdef RADIO_DEBUG
/*
		if (call DiagMsg.record()) {
			call DiagMsg.str("EPR"); 
			call DiagMsg.send();
		}
*/ 
#endif
	
	}

	void do_wait_for_tx();
	void do_listen();
	void do_disable_reception();


	void do_wait_for_first_byte() {
		atomic {
			rxstate = RX_WAITING_FOR_FIRST_BYTE;
			waitforradio = TRUE;
			releaseControl(RX_CONTROLLED);	
		}
		TOSH_SET_LED_3_PIN();
		TOSH_CLR_LED_1_PIN();
		do_pending_thing();
/*
		if (txpending) {
			if (getControl(TX_CONTROLLED)) { // ==SUCCESS always
				if (mode & TX_WAIT_ENABLED_BIT) {
					do_wait_for_tx();
					releaseControl(TX_CONTROLLED);
				} else if (mode & TX_LISTEN_ENABLED_BIT) {
					do_listen();
				} else {
					do_disable_reception();
				}
			}
		}
*/ 
	}

	async event result_t IA4420Interrupt.FFIT() 
	{
		uint8_t local = 0;
		atomic {
			if (rxstate == RX_WAITING_FOR_FIRST_BYTE && waitforradio) {
				local = 1;

				if (!getControl(RX_CONTROLLED)) {
					fatalerror();
					local = 4;
				}

				TOSH_CLR_LED_3_PIN();
				TOSH_SET_LED_1_PIN();
				waitforradio = FALSE;
			}
			else if (rxstate == RX_WAITING_FOR_NEXT_BYTE && waitforradio) {
				local = 2;
				waitforradio = FALSE;
			}
		}
#ifdef RADIO_DEBUG
		if (local==1)
			call Timer.start(TIMER_ONE_SHOT, 1000);
#endif		
		if (local==1 || local==2)
			do_read_fifo();

		else //if (!(good && post do_next_rx_task()))
		{

#ifdef RADIO_DEBUG
			if (call DiagMsg.record()) {
				call DiagMsg.str("FFIT error! state="); 
				call DiagMsg.uint8(rxstate); 
				call DiagMsg.uint8(txstate); 
				call DiagMsg.uint8((uint8_t)(rxbyteptr-((uint8_t *) rxbufptr)));
				call DiagMsg.send();
			}
#endif
		}

/*
		if (call DiagMsg.record()) {
			call DiagMsg.str("FFIT "); 
			call DiagMsg.uint8(rxstate); 
			call DiagMsg.uint8(txstate); 
			call DiagMsg.uint8(local);
			call DiagMsg.uint8((uint8_t)(rxbyteptr-((uint8_t *) rxbufptr)));
			call DiagMsg.send();
		}
*/
		
		return SUCCESS;
	}

	void do_read_fifo() {
		atomic {
			rxstate = RX_READING_FIFO;
		}	
		call HPLIA4420Command.RFRC(); 
	}

	/** On receive FIFO read done:  **/
	async event result_t HPLIA4420Command.RFRCDone(uint8_t data) {
		uint8_t rxs; //lstate, lsubstate;
		atomic rxs = rxstate;
		if (rxs != RX_READING_FIFO)
			return SUCCESS;

		(*rxbyteptr) = data; // writing data
		
		if ((rxbyteptr == (uint8_t *)(&(rxbufptr->length))) && (rxbufptr->length > TOSH_DATA_LENGTH)) {

#ifdef RADIO_DEBUG
			if (call DiagMsg.record()) {
				call DiagMsg.str("lgth err! st="); 
				call DiagMsg.uint8(rxstate); 
				call DiagMsg.uint8(txstate); 
				call DiagMsg.uint8(txstate); 
				call DiagMsg.uint8(waitforradio); 
				call DiagMsg.send();
			}
#endif
			do_on_rx_error();
			return SUCCESS;

//			rxactive = FALSE;
//			waitforradio = FALSE;
//			do_next_rx(); //post do_next_rx_task();
		}
		else
		{
			rxbyteptr++;

			if (rxbyteptr <= (uint8_t *)(&(rxbufptr->data)) + 
				rxbufptr->length) { // crc not received yet
				crc = crcByte(crc, data);
				
			}

			if (rxbyteptr > (uint8_t*)(&(rxbufptr->length))) // length received
			{ 
				if (rxbyteptr == (uint8_t *)(&(rxbufptr->data)) + 
					rxbufptr->length + 2) {
					uint16_t reccrc = *((uint16_t *)(rxbyteptr-2));
					rxbufptr->crc = reccrc; // copying crc
/*
#ifdef RADIO_DEBUG
			if (call DiagMsg.record()) {
				call DiagMsg.str("crc="); 
				call DiagMsg.uint16(crc); 
				call DiagMsg.uint16(reccrc); 
				call DiagMsg.send();
			}
#endif
*/
					if (crc==rxbufptr->crc)
						rxbufptr->crc = 1;
					else
						rxbufptr->crc = 0;


		// for testing:
//					rxbufptr->crc = 1;

					do_finish_packet_receive();
					return SUCCESS;
				}
				
			}			
			do_wait_for_next_byte();
		}

 		return SUCCESS;
	}

	void do_wait_for_next_byte() {
		atomic {
			rxstate = RX_WAITING_FOR_NEXT_BYTE;
			waitforradio = TRUE;
		}	
	}
	
	void do_finish_packet_receive() {
//		atomic {
//			packetreceivedpending = TRUE;
//		}
//		post PacketRcvd();
		do_disable_pattrecog2();
	}

	void do_disable_pattrecog2() {
		atomic {
			waitforradio = FALSE;
			rxstate = RX_DISABLING_PATTERN_RECOG2;
		}
#ifdef RADIO_DEBUG

		if (call DiagMsg.record()) {
			call DiagMsg.str("DPR2"); 
			call DiagMsg.send();
		}

#endif
		if (!call HPLIA4420Command.FRMC(0x80)) // disable pattern recog
			fatalerror();
	}


	void onCommandDoneRX() {
		uint8_t local;
		switch (rxstate) {
			case RX_ENABLING:
				do_disable_pattrecog();
				break;
			case RX_DISABLING_PATTERN_RECOG:
				local = 0;

				//wait();

				atomic {

//					if (packetreceivedpending) {
//						receivestartpending = 1;
//						local=1;
//					}

//					if (txpending) {
//						receiving = FALSE;
//						local += 2;
//					}
				}

//				if (local&2)
//					do_wait_for_tx();

//				if (local == 0)
					do_enable_pattrecog();
				break;

			case RX_ENABLING_PATTERN_RECOG:
				do_wait_for_first_byte();
				break;

			case RX_DISABLING_PATTERN_RECOG2:
//				atomic {
//					packetreceivedpending = TRUE;
//				}
				post PacketRcvd();

				//do_disable_pattrecog();
				break;

			case RX_ERROR:
				// Tesztelni kell!
				do_disable_pattrecog();
				break;
		}
	}

/*
	task void trace() {
#ifdef RADIO_DEBUG
 		if (call DiagMsg.record()) {
			call DiagMsg.str("Send.send()"); 
			call DiagMsg.send();
		}
#endif
	}
*/

 /**********************************************************
   * Send
   *    
   **********************************************************/
	void do_wait_for_tx();
	void do_listen();
	void do_disable_reception();
	void do_write_preamble();
	void do_send_sync1();
	void do_send_sync2();
	void do_start_transmission();
	void do_send_byte();
	void do_send_dummy();
	void do_disable_transmission();
	void do_enable_reception();
	task void PacketSent();

	void sendFailed() {
		signal Send.sendDone(txbufptr, FAIL);
	}
 	
	command result_t Send.send(TOS_MsgPtr pMsg) {
		uint8_t res = 0;
 /*
 		if (call DiagMsg.record()) {
			call DiagMsg.str("Send.send()"); 
			call DiagMsg.send();
		}
*/
//		post trace();		
		atomic {
			if (txstate != TX_NO_PACKET_PENDING)
				res = 0;
			else {
				txstate = TX_PACKET_PENDING;
				txpending = TRUE;
				txstartpending = TRUE;
				txbufptr = pMsg;


//				pMsg->addr = toLSB16(pMsg->addr);

// For debugging
//				if (!TOSH_READ_GPIO_1_PIN()) // button1
//					pMsg->length = 32;
//				if (getControl(TX_CONTROLLED))
					res = 1;
//				else
//					res = 2;
	
			}
		}

		if (res==0) {
			//sendFailed();
			return FAIL;
		}

		if (res==1) {
			do_tx_pending();
/*			if (mode & TX_WAIT_ENABLED_BIT) {
				do_wait_for_tx();
				releaseControl(TX_CONTROLLED);
			} else if (mode & TX_LISTEN_ENABLED_BIT) {
				do_listen();
			} else {
				do_disable_reception();
			}
*/
		}

#ifdef RADIO_DEBUG
			
		call Timer.start(TIMER_ONE_SHOT, 1000);

 		if (call DiagMsg.record()) {
			call DiagMsg.str("Send.send()"); 
			call DiagMsg.uint8(rxstate); 
			call DiagMsg.uint8(txstate); 
			call DiagMsg.send();
		}
#endif
		return SUCCESS;
	}


	bool do_tx_pending() {
		if (txstartpending) {
			if (getControl(TX_CONTROLLED)) {
				txstartpending = FALSE;
				if (mode & TX_WAIT_ENABLED_BIT) {
					do_wait_for_tx();
					releaseControl(TX_CONTROLLED);
					do_pending_thing();
				} else if (mode & TX_LISTEN_ENABLED_BIT) {
					do_listen();
				} else {
					do_disable_reception();
				}
				return TRUE;
			}
		}
		return FALSE;
	}

	void do_wait_for_tx() {
		atomic {
			txstate = TX_WAITING;
		}
		call TimerJiffyAsync.setOneShot(((call Random.rand() & 0x0F) + 10)*10);
	}

	async event result_t TimerJiffyAsync.fired() {
		uint8_t res;
		atomic {
			if (txstate != TX_WAITING){
				res = 0;
			} else if (!getControl(TX_CONTROLLED)) {
				txstate = TX_PACKET_PENDING;
				txstartpending = TRUE;
				res = 1;
			} else {
				res = 2; // tx has control
			}
		}
		if (res==2)
			if (mode & TX_LISTEN_ENABLED_BIT)
				do_listen();
			else
				do_disable_reception();
		return SUCCESS;
	}
	
	// pre: mode == TX_CONTROLLED
	void do_listen() {
		atomic txstate = TX_LISTENING;
		call HPLIA4420Command.SRC();
	}

	async event result_t HPLIA4420Command.SRCDone(uint16_t status) {

#ifdef RADIO_DEBUG
			if (call DiagMsg.record()) {
				call DiagMsg.str("status="); 
				call DiagMsg.uint16(status); 
				call DiagMsg.uint16(status & (1<<RSSI)); 
				call DiagMsg.send();
			}
#endif
		if (txstate == TX_LISTENING)
		{
			if (status & (1<<RSSI)) {
				do_wait_for_tx();
				releaseControl(TX_CONTROLLED);
				do_pending_thing();
			} else {
				do_disable_reception();
			}
 		}
		return SUCCESS;
	}

	void inittxpuffer() {
		byteptr = (uint8_t *) txbufptr;
		endbyteptr = ((uint8_t *)txbufptr) + TOSH_HEADER_LENGTH + txbufptr->length;

		// Calculating CRC			
		txbufptr->crc=0;
		while (byteptr != endbyteptr)
			txbufptr->crc = crcByte(txbufptr->crc, *(byteptr++));

		byteptr = (uint8_t *) txbufptr;
		*((uint16_t *)endbyteptr) = txbufptr->crc;
		endbyteptr+=sizeof(txbufptr->crc);		
	}
	
	void do_disable_reception() {
		atomic {
			rxstate = RX_SUSPENDED;
			txstate = TX_DISABLING_RECEIVER;
			debug_b=1;
		}
		TOSH_SET_LED_2_PIN();
		call HPLIA4420Command.PMC(0x18); // disable receiver & transmitter
	}
	
	void do_write_preamble1() {
		atomic {
			txstate = TX_WRITING_PREAMBLE1;
		}
			call HPLIA4420Command.TRWC(0xAA); // Write preamble
	}

	void do_write_preamble2() {
		atomic {
			txstate = TX_WRITING_PREAMBLE2;
		}
			call HPLIA4420Command.TRWC(0xAA); // Write preamble
	}

	void do_write_preamble3() {
		atomic {
			txstate = TX_WRITING_PREAMBLE3;
		}
			call HPLIA4420Command.TRWC(0xAA); // Write preamble
	}

	void do_write_preamble4() {
		atomic {
			txstate = TX_WRITING_PREAMBLE4;
		}
			call HPLIA4420Command.TRWC(0xAA); // Write preamble
	}
	
	void do_start_transmission() {
		atomic txstate = TX_STARTING_TRANSMISSION;
		call HPLIA4420Command.PMC(0x38); // enable transmitter
	}

	void onCommandDoneTX() {
		uint8_t txs;
		atomic txs = txstate;
		switch (txs) {
			case TX_DISABLING_RECEIVER:
				debug_b|=2;
				inittxpuffer();
				debug_b|=4;
				do_write_preamble1();
				break;
			case TX_STARTING_TRANSMISSION:
				atomic {
					txstate = TX_WRITING_PREAMBLE3;
					waitforradio = TRUE;
				}	
				break;
			case TX_ENABLING_RECEIVER:
				break;
			case TX_DISABLING_TRANSMITTER:
				post PacketSent();
//				do_enable_reception();
				break;
		}
	}

	void do_wait_txreg_empty() {
		atomic {
			txstate = TX_SENDING;
			waitforradio = TRUE;
		}	
	}

	async event result_t IA4420Interrupt.RGIT() {
		uint8_t local = 0;
//		TOSH_SET_LED_1_PIN();
		atomic {
			if (!waitforradio)
				local = 0;
			else {
				waitforradio = FALSE;
				switch (txstate) {
					case TX_WRITING_PREAMBLE3:
							local = 6;
						break;
					case TX_WRITING_PREAMBLE4:
							local = 7;
						break;
					case TX_SENDING_SYNC1:
							local = 4;
						break;
					case TX_SENDING_SYNC2:
							local = 5;
						break;
					case TX_SENDING:
						if (byteptr != endbyteptr)
							local = 1;
						else
							local = 2;
						break;
					case TX_SENDING_DUMMY:
						local = 3;
						break;

					case TX_DISABLING_RECEIVER:
						local = 8;
						break;
				}
			}
		}

		switch (local) {		
			case 6:
				do_write_preamble3();
				break;
			case 7:
				do_write_preamble4();
				break;
			case 4:
				do_send_sync1();
				break;
			case 5:
				do_send_sync2();
				break;
			case 1:
				do_send_byte();
				break;
			case 2:
				do_send_dummy();
				break;
			case 3:
				do_disable_transmission();
//				do_enable_reception();
				break;

			case 8:
#ifdef RADIO_DEBUG
				if (call DiagMsg.record()) {
					call DiagMsg.str("RGIT st="); 
					call DiagMsg.uint8(rxstate); 
					call DiagMsg.uint8(txstate); 
					call DiagMsg.uint8(debug_b); 
					call DiagMsg.send();
				}
#endif				
				break;

			default:
#ifdef RADIO_DEBUG
			if (call DiagMsg.record()) {
				call DiagMsg.str("RGIT error! state="); 
				call DiagMsg.uint8(rxstate); 
				call DiagMsg.uint8(txstate); 
				call DiagMsg.uint8((uint8_t)(rxbyteptr-((uint8_t *) rxbufptr)));
				call DiagMsg.send();
			}
#endif
		}
//		TOSH_CLR_LED_1_PIN();
		return SUCCESS;
	}

	void do_send_sync1() {
		atomic txstate = TX_SENDING_SYNC1; // ?
		call HPLIA4420Command.TRWC(0x2D);
	}

	void do_send_sync2() {
		atomic txstate = TX_SENDING_SYNC2;
		call HPLIA4420Command.TRWC(0xD4);
	}

	void do_send_byte() {
		uint8_t txbyte;
		atomic {
			txstate = TX_SENDING;
			txbyte = *(byteptr++);
		}
		call HPLIA4420Command.TRWC(txbyte);
	}

	void do_send_dummy() {
		atomic txstate = TX_SENDING_DUMMY;
		call HPLIA4420Command.TRWC(0);
	}
	

	void do_disable_transmission() {
		atomic {
			txstate = TX_DISABLING_TRANSMITTER;
		}
		call HPLIA4420Command.PMC(0x18); // disable receiver & transmitter

	}	

	void do_enable_reception() {
//		atomic {
//			txstate = TX_ENABLING_RECEIVER;
//		}
		releaseControl(TX_CONTROLLED);
		do_enable_rx();
		//post PacketSent();
	}


	task void PacketSent() {
		signal Send.sendDone(txbufptr, SUCCESS);
		TOSH_CLR_LED_2_PIN();
		atomic {
			txstate = TX_NO_PACKET_PENDING;
			txpending = FALSE;
		}
#ifdef RADIO_DEBUG

		if (call DiagMsg.record()) {
			call DiagMsg.str("Packet sent!"); 
			call DiagMsg.send();
		}
		if (call DiagMsg.record()) {
			call DiagMsg.str("Tout!"); 
			call DiagMsg.uint8(rxstate); 
			call DiagMsg.uint8(txstate); 
			call DiagMsg.uint8(waitforradio); 
//			call DiagMsg.uint8(receiving); 
			call DiagMsg.uint8(packetreceivedpending); 
			call DiagMsg.uint8(receivestartpending); 
			call DiagMsg.uint8(txpending); 
			call DiagMsg.uint8(lastcommand); 
//			call DiagMsg.uint8((uint8_t)(rxbyteptr-((uint8_t *) rxbufptr)));
			call DiagMsg.send();
		}

#endif

		do_enable_reception();
	}

	async event result_t HPLIA4420Command.TRWCDone() {
		uint8_t txs;
		atomic txs = txstate;
		switch (txs) {
			case TX_WRITING_PREAMBLE1:
				do_start_transmission();
//				do_write_preamble2();
				break;
			case TX_WRITING_PREAMBLE2:
				do_start_transmission();
				break;
			case TX_WRITING_PREAMBLE3:
				atomic {
					txstate = TX_WRITING_PREAMBLE4;
					waitforradio = TRUE;
				}	
				break;
			case TX_WRITING_PREAMBLE4:
				atomic {
					txstate = TX_SENDING_SYNC1;
					waitforradio = TRUE;
				}	
				break;
			case TX_SENDING_SYNC1:
				atomic {
					txstate = TX_SENDING_SYNC2;
					waitforradio = TRUE;
				}	
				break;
			case TX_SENDING_SYNC2:
				atomic {
					txstate = TX_SENDING;
					waitforradio = TRUE;
				}	
				break;
			case TX_SENDING:
				atomic {
					txstate = TX_SENDING;
					waitforradio = TRUE;
				}	
				break;
			case TX_SENDING_DUMMY:
				atomic {
					txstate = TX_SENDING_DUMMY;
					waitforradio = TRUE;
				}	
				break;
		}
		return SUCCESS;
	}
	
	

/** IA4420Interrupt handlers  **/


	async event result_t IA4420Interrupt.POR() {
		return post inittask();
	}

	async event result_t IA4420Interrupt.RGUR() {
#ifdef RADIO_DEBUG
		if (call DiagMsg.record()) {
			call DiagMsg.str("RGUR! state="); 
			call DiagMsg.uint8(rxstate); 
			call DiagMsg.uint8(txstate); 
			call DiagMsg.uint8((uint8_t)(rxbyteptr-((uint8_t *) rxbufptr)));
			call DiagMsg.send();
		}
#endif
		return SUCCESS;
	}

	async event result_t IA4420Interrupt.FFOV() {
#ifdef RADIO_DEBUG
		if (call DiagMsg.record()) {
			call DiagMsg.str("FFOV! state="); 
			call DiagMsg.uint8(rxstate); 
			call DiagMsg.uint8(txstate); 
			call DiagMsg.uint8((uint8_t)(rxbyteptr-((uint8_t *) rxbufptr)));
//			call DiagMsg.uint8(receiving); 
			call DiagMsg.send();
		}
#endif
//		waitforradio = FALSE;
//		do_next_rx();
		
		return SUCCESS;
	}

	async event result_t IA4420Interrupt.WKUP() {
		return SUCCESS;
	}

	async event result_t IA4420Interrupt.EXT() {
		return SUCCESS;
	}

	async event result_t IA4420Interrupt.LBD() {
		return SUCCESS;
	}
/*
	event result_t Debug.outputComplete(result_t success) {
		return SUCCESS;
	}
*/
	void onCommandDoneControl();

	/** Radio command has been sent:  **/
	async event result_t HPLIA4420Command.commandDone() {
//		uint8_t lstate, lsubstate;
		//atomic {
//			lstate = state;
		//}

		if (rxstate == RX_INIT) {
			atomic sync_done = 1;
			return SUCCESS;
		}

		onCommandDoneControl();

		onCommandDoneRX();
		onCommandDoneTX();
		return SUCCESS;
	}

#ifdef RADIO_DEBUG

// Time out
	event result_t Timer.fired() {
		atomic{
			if (call DiagMsg.record()) {
				call DiagMsg.str("Tout!"); 
				call DiagMsg.uint8(rxstate); 
				call DiagMsg.uint8(txstate); 
				call DiagMsg.uint8(waitforradio); 
//				call DiagMsg.uint8(receiving); 
				call DiagMsg.uint8(packetreceivedpending); 
				call DiagMsg.uint8(receivestartpending); 
				call DiagMsg.uint8(txpending); 
				call DiagMsg.uint8(lastcommand); 
//				call DiagMsg.uint8((uint8_t)(rxbyteptr-((uint8_t *) rxbufptr)));
				call DiagMsg.send();
			}
		}

//		call HPLIA4420Command.SRC();
		TOSH_SET_LED_3_PIN();
		return SUCCESS;
	}
#endif

	bool do_control_pending() {
		if (control_pending) {
			if (getControl(CON_CONTROLLED)) {
				if (ctrlstate & FREQ_DIRTY) {
					control_executing = TRUE;
					call HPLIA4420Command.FSC(newfreq);
					ctrlstate &= ~FREQ_DIRTY;
				} else if (ctrlstate & DATARATE_DIRTY) {
					control_executing = TRUE;
					call HPLIA4420Command.DRC(newdatarate);
					ctrlstate &= ~DATARATE_DIRTY;
				} else if (ctrlstate & TXPOWER_DIRTY) {
					control_executing = TRUE;
					call HPLIA4420Command.SetTXPower(newtxpower);
					ctrlstate &= ~TXPOWER_DIRTY;
				}
				return TRUE;
			}
		}
		return FALSE;				
	}

	command result_t IA4420Control.SetFrequency(uint16_t freq) {
		result_t res;
		atomic {
			if (ctrlstate & FREQ_DIRTY)
				res = FAIL;
			else {
				ctrlstate |= FREQ_DIRTY;
				control_pending = TRUE;
				newfreq=freq;
				res = SUCCESS;
			}
		}
		if (res)
			do_control_pending();
		return res;
	}

	command result_t IA4420Control.SetDataRate(uint8_t rate) {
		result_t res;
		atomic {
			if (ctrlstate & DATARATE_DIRTY)
				res = FAIL;
			else {
				ctrlstate |= DATARATE_DIRTY;
				control_pending = TRUE;
				newdatarate=rate;
				res = SUCCESS;
			}
		}
		if (res)
			do_control_pending();
		return res;
	}

	command result_t IA4420Control.SetTXPower(uint8_t power) {
		result_t res;
		atomic {
			if (ctrlstate & TXPOWER_DIRTY)
				res = FAIL;
			else {
				ctrlstate |= TXPOWER_DIRTY;
				control_pending = TRUE;
				newtxpower = power;
				res = SUCCESS;
			}
		}
		if (res)
			do_control_pending();
		return res;
	}

/*
	command result_t IA4420Control.SetBandwidth() {
	}

	command result_t IA4420Control.SetDeviation() {
	}
*/
/*
	async command result_t IA4420Control.TxMode();

	async command result_t IA4420Control.RxMode();

	async command result_t IA4420Control.IdleMode();
*/

//	command result_t IA4420Control.SetRXSensitivity();	

/*
	command result_t IA4420Control.SetWakeupTimer();

	command result_t IA4420Control.SetLowBattery();

	command result_t IA4420Control.SetClockDivider();
*/
	void onCommandDoneControl() {
		if (!control_executing) 
			return;
		control_executing = FALSE;
		control_pending = (ctrlstate!=0);
		releaseControl(CON_CONTROLLED);
		do_pending_thing();
	}

}
