/**
 * @author Laszlo Siroki
 */

includes byteorder;
includes crc;
includes IA4420;

module IA4420RadioM {
	provides {
		interface StdControl;
		interface BareSendMsg as Send;
		interface ReceiveMsg as Receive;
	}
	uses {
		//interface IA4420Spi;

		interface HPLIA4420Command;
		interface IA4420Interrupt;

		interface Random;

#ifdef RADIO_DEBUG
//		interface IntOutput as Debug;
		interface DiagMsg;
#endif		
		interface TimerJiffyAsync;
	
	}
}

implementation {

//uint8_t transfer_setting = 0x00;

enum {
    DISABLED_STATE = 0,
	INIT_STATE,
	IDLE_STATE,
	TX_STATE,
	RX_STATE,
//	PRE_TX_STATE,
	
	TX_IDLE = 0,
	TX_PENDING,
	TX_WAITING,
	TX_LISTENING,
	TX_TRANSMITTING
};

	uint8_t state;
	uint8_t substate;

//	uint8_t prev_state;

	TOS_MsgPtr txbufptr;  // pointer to transmit buffer
	TOS_MsgPtr rxbufptr;  // pointer to receive buffer
	TOS_Msg RxBuf;	// save received messages
	uint16_t crc;
	
	uint8_t *rxbyteptr;
	
	bool waitforradio = FALSE;
	
	uint8_t txstate = TX_IDLE;

	bool rxactive = FALSE;

	
	volatile uint8_t sync_done;

/*********************
*   Local functions  *
*********************/
/*
void debugledon() {
			TOSH_MAKE_GPIO_15_OUTPUT();
			TOSH_SET_GPIO_15_PIN();
}
*/
/*
void wait() {
	uint32_t i;
	for(i=0;i<20000000;i++) ;
}
*/

void wait_for_spi() {
	uint8_t lsync_done = 0;

	while (lsync_done == 0)
		atomic {lsync_done = sync_done;}

	atomic {
		sync_done = 0;
	}

}

#ifdef RADIO_DEBUG

	task void valami() {
		if (call DiagMsg.record()) {
			call DiagMsg.str("hiba!"); 
			call DiagMsg.send();
		}
	}
#endif

/********************************/
/** init/start/stop functions  **/
/********************************/

	task void inittask();
	result_t rxMode();
	
	command result_t StdControl.init() {
		TOSH_MAKE_IA4420_NSEL_OUTPUT();	// nSEL pin
		TOSH_MAKE_IA4420_NIRQ_INPUT();  // nIRQ pin
		
		TOSH_MAKE_GPIO_14_INPUT(); // nIRQ Connected with this pin also in current implementation

		call Random.init();

		atomic {
			state = DISABLED_STATE;
			rxbufptr = &RxBuf;
		}
		// Setting up interrupt : at first interrupt on low level
		cbi(EIMSK, INT5);
		cbi(EICRB, ISC50);
		cbi(EICRB, ISC51);
		sbi(EIMSK, INT5);
		// Radio initialisation will be done by POR interrupt.
		return SUCCESS;
	}

// Start and stop are useless now
	command result_t StdControl.start() {
		return SUCCESS;
	}
  
	command result_t StdControl.stop() {
//		atomic state = DISABLED_STATE;
		return SUCCESS;
	}

	void com_r04_s01__Init_433 (unsigned int sub_frequency, uint8_t baseband_bw, uint8_t deviation, uint8_t power, uint8_t data_rate)
	{
		uint8_t rssitreshold = 4;
		atomic {
			state = INIT_STATE;
			sync_done = 0;
		}

		// Configuration Setting command
		call HPLIA4420Command.CSC(0xD7);// enables the internal data register, rx fifo, 433 MHz, 12 pF
		wait_for_spi();

		// Configuration Setting command again experiences shows that it should be done twice
		call HPLIA4420Command.CSC(0xD7);// enables the internal data register, rx fifo, 433 MHz, 12 pF
		wait_for_spi();
	  
		// Sub frequencies
		call HPLIA4420Command.FSC(sub_frequency);
		wait_for_spi();

		// Power management command
		call HPLIA4420Command.PMC(0x18);
		wait_for_spi();
		
		// Setting baseband BW and RSSI treshold
		call HPLIA4420Command.RCC(0x0600 | baseband_bw | rssitreshold);
		wait_for_spi();

		// Transmiter Control command
		call HPLIA4420Command.TXCCC(deviation|power);
		wait_for_spi();

		// Data rate command
		call HPLIA4420Command.DRC(data_rate);
		wait_for_spi();
	}


// Initialisation task: currently non-split-phase
	task void inittask() {
		com_r04_s01__Init_433(
			IA4420_DEFAULT_FREQ,
			IA4420_DEFAULT_BASEBAND_BW,
			IA4420_DEFAULT_FSK_DEVIATION,
			IA4420_DEFAULT_OUTPUT_POWER,
			IA4420_DEFAULT_DATA_RATE
			);

		atomic state = IDLE_STATE;

#ifdef RADIO_DEBUG
		if (call DiagMsg.record()) {
			call DiagMsg.str("Init"); 
			call DiagMsg.uint16(TOS_LOCAL_ADDRESS); 
			call DiagMsg.send();
		}
#endif

		rxMode();

#ifdef RADIO_DEBUG
		if (call DiagMsg.record()) {
			call DiagMsg.str("rx on"); 
			call DiagMsg.send();
		}
#endif

	}



/***********************
*   Receive functions  *
************************/

	void do_next_rx();
	void begintxwait();
	
/** Change to RX mode  **/

	result_t rxMode() {
		uint8_t lstate;
		result_t ret = FAIL;
		atomic {
			lstate = state;
			if (lstate == IDLE_STATE) {
				state = RX_STATE;
				substate=0;
			}
			if(state == RX_STATE)
				ret = SUCCESS;
		}

		if (lstate == IDLE_STATE)		
			do_next_rx();

		return ret;
	}

/** Signal packet reception **/

	task void PacketRcvd() {
		TOS_MsgPtr pBuf;
		uint8_t lgt;
		
		atomic {
			pBuf = rxbufptr;
		}
		lgt = pBuf->length;
		pBuf = signal Receive.receive((TOS_MsgPtr)pBuf);
		
		atomic {
			if (pBuf)
				rxbufptr = pBuf;
		}

		substate = 1;
		do_next_rx();

		rxactive = FALSE;
		if (txstate==TX_PENDING)	// If transmission pending, begin wait&listen
			begintxwait();

#ifdef RADIO_DEBUG
		
		if (call DiagMsg.record()) {
			call DiagMsg.str("Packet received!"); 
			call DiagMsg.int8(lgt); 
			call DiagMsg.send();
		}
#endif
	}


/** Do next RX action  **/
	void do_next_rx() {
		uint8_t lsubstate;
	
		atomic lsubstate=substate;
		switch(lsubstate)
		{
			// Switch on receiver
			case 0:
				rxactive = FALSE;
				call HPLIA4420Command.PMC(0xD8); // RX mode
				break;
	
			case 1:
				rxactive = FALSE;
				atomic rxbyteptr = (uint8_t *) rxbufptr;
				call HPLIA4420Command.FRMC(0x80); // disable pattern recog
				break;

			case 2:
				atomic crc=0;
				rxactive = FALSE;
				call HPLIA4420Command.FRMC(0x82); // enable pattern recog
				break;
	
			case 3: // receiving packet
				rxactive = TRUE;
				call HPLIA4420Command.RFRC(); 
				break;
				
			case 4:
			case 5:
				call HPLIA4420Command.RFRC(); 
				break;
	
			case 6:
				call HPLIA4420Command.FRMC(0x80); // disable pattern recog
//				rxactive = FALSE;
//				if (txstate==TX_PENDING)	// If transmission pending, begin wait&listen
//					begintxwait();
				break;		
		}
	}


	/** On receive FIFO read done:  **/
	async event result_t HPLIA4420Command.RFRCDone(uint8_t data) {
		uint8_t lstate, lsubstate;
		
		atomic {
			lstate = state;
			lsubstate = substate;
		}

		if (lstate==RX_STATE) {
			switch(lsubstate) {
				case 3: // receive data

					(*rxbyteptr) = data;
					crc = crcByte(crc, data);
					
					if ((rxbyteptr == (uint8_t *)(&(rxbufptr->length))) && (rxbufptr->length > TOSH_DATA_LENGTH)) {

#ifdef RADIO_DEBUG
						if (call DiagMsg.record()) {
							call DiagMsg.str("lgth err! st="); 
							call DiagMsg.uint8(state); 
							call DiagMsg.uint8(substate); 
							call DiagMsg.uint8(txstate); 
							call DiagMsg.uint8(waitforradio); 
							call DiagMsg.send();
						}
#endif
						substate = 0;
						rxactive = FALSE;
						waitforradio = FALSE;
						do_next_rx(); //post do_next_rx_task();

						break; 						 
					}
					else
					{
						rxbyteptr++;
						if (rxbyteptr > (uint8_t*)(&(rxbufptr->length)) && 
							rxbyteptr == (uint8_t *)(&(rxbufptr->data)) + rxbufptr->length )// crc
						{
							substate = 4;
						}
						
						waitforradio = TRUE;
					}
					break;
				
				case 4: //CRCL
					rxbufptr->crc = data;
					substate = lsubstate+1;
					waitforradio = TRUE;

					break;
				case 5: //CRCH
					rxbufptr->crc += ((uint16_t)data)<<8;

					substate = 6;


					if (crc==rxbufptr->crc)
						rxbufptr->crc = 1;
					else
						rxbufptr->crc = 0;
		// for testing:
//					rxbufptr->crc = 1;

					do_next_rx();
					break;
			}
		}
		return SUCCESS;
	}


 /**********************************************************
   * Send
   * - Xmit a packet
   *    USE SFD FALLING FOR END OF XMIT !!!!!!!!!!!!!!!!!! interrupt???
   * - If in power-down state start timer ? !!!!!!!!!!!!!!!!!!!!!!!!!s
   * - If !TxBusy then 
   *   a) Flush the tx fifo 
   *   b) Write Txfifo address
   *    
   **********************************************************/

	command result_t Send.send(TOS_MsgPtr pMsg) {
		result_t res = FAIL;
		
		atomic {
			if (txstate != TX_IDLE || (state != RX_STATE))
				res = FAIL;
			else {
				txstate = TX_PENDING;
				txbufptr = pMsg;
//				pMsg->addr = toLSB16(pMsg->addr);

// For debugging
//				if (!TOSH_READ_GPIO_1_PIN()) // button1
//					pMsg->length = 32;

				res = SUCCESS;
			}
		}
		if (!res)
			return res;

		if (!rxactive)
			begintxwait();
/*
#ifdef RADIO_DEBUG
 		if (call DiagMsg.record()) {
			call DiagMsg.str("Send.send()"); 
			call DiagMsg.uint8(state); 
			call DiagMsg.uint8(substate); 
			call DiagMsg.uint8(txstate); 
			call DiagMsg.send();
		}
#endif
*/		  
		return SUCCESS;
	}

/*
	inline bool rxwaiting(){
		return (!rxactive);
	}
*/

/** Do next TX action  **/
void do_next_tx() {
	uint8_t lsubstate;
	uint8_t lstate;
	static uint8_t *byteptr, *endbyteptr;
	atomic {
		lstate=state;
		lsubstate=substate;
	}
	switch(lsubstate)
	{
		case 0:
			byteptr = (uint8_t *) txbufptr;
			endbyteptr = ((uint8_t *)txbufptr) + TOSH_HEADER_LENGTH + txbufptr->length;

			// Calculating CRC			
			txbufptr->crc=0;
			while (byteptr != endbyteptr)
				txbufptr->crc = crcByte(txbufptr->crc, *(byteptr++));

			byteptr = (uint8_t *) txbufptr;
				
			call HPLIA4420Command.PMC(0x18); // disable transmitter
			break;
			
		case 1:
			call HPLIA4420Command.TRWC(0xAA); // Write preamble
			break;
			
		case 2:
			call HPLIA4420Command.TRWC(0xAA); // Write preamble
			break;

		case 3:
			call HPLIA4420Command.PMC(0x38); // enable transmitter
			break;
			
		case 4:
		case 5:
		case 6:
		case 7:
		case 8:
		case 9:
			call HPLIA4420Command.TRWC(0xAA); // Write preamble
			break;
		
		/*** Sending syncron pattern ***/	
		case 10:
			call HPLIA4420Command.TRWC(0x2D);
			break;
		case 11:
			call HPLIA4420Command.TRWC(0xD4);
			break;

		case 12:
			if (byteptr != endbyteptr) {
				call HPLIA4420Command.TRWC(*byteptr);
				byteptr++;
			} else {
//				atomic substate=53; // Only for testing!!!
//				call IA4420Spi.sendCommand(0x82, 0x18); // disable transmitter

				atomic substate=50;
				call HPLIA4420Command.TRWC((uint8_t)(txbufptr->crc & 0xFF));
//				com_r04_s01__Write_Transmit_data (0);
			}
			break;

		case 51:
			call HPLIA4420Command.TRWC((uint8_t)(txbufptr->crc >> 8));
//			com_r04_s01__Write_Transmit_data (0);
			break;

		case 52:
			call HPLIA4420Command.TRWC(0);  //Dummy byte
			break;
			
		case 53:
//			if (prev_state == RX_STATE)
				call HPLIA4420Command.PMC(0xD8); // disable transmitter, enable receiver
/*			else 
				call HPLIA4420Command.PMC(0x18); // disable transmitter
*/
			break;
			
	}

}

/*
	task void do_next_tx_task() {
		do_next_tx();
	}
*/

	void begintxwait() {
		txstate = TX_WAITING;
		call TimerJiffyAsync.setOneShot((call Random.rand() & 0x1F) + 1);
	}


	void starttransmission() {
		txstate = TX_TRANSMITTING;
		state = TX_STATE;
		substate = 0;
		do_next_tx();
	}


	async event result_t TimerJiffyAsync.fired() {
		if (txstate!=TX_WAITING){
			return SUCCESS;
		} else if (rxactive) {
			txstate = TX_PENDING;
			return SUCCESS;
		} else {
			//listen to channel
			txstate = TX_LISTENING;
			call HPLIA4420Command.SRC();
		}
	}

	async event result_t HPLIA4420Command.SRCDone(uint16_t status) {
		if (rxactive) {
			txstate = TX_PENDING;
			return SUCCESS;
		}
		if (txstate == TX_LISTENING)
		{

#ifdef RADIO_DEBUG

			if (call DiagMsg.record()) {
				call DiagMsg.str("status="); 
				call DiagMsg.uint16(status); 
				call DiagMsg.uint16(status & (1<<RSSI)); 
				call DiagMsg.send();
			}
#endif

			if (status & (1<<RSSI))
				begintxwait();
			else
				starttransmission();
 
		}
		return SUCCESS;
	}

	task void signalsenddone() {
//		txstate = TX_IDLE;
		signal Send.sendDone(txbufptr, SUCCESS);
#ifdef RADIO_DEBUG
		if (call DiagMsg.record()) {
			call DiagMsg.str("Packet sent!"); 
			call DiagMsg.send();
		}
#endif
	}


/** IA4420Interrupt handlers  **/
	async event result_t IA4420Interrupt.FFIT() 
	{
		uint8_t good;
		atomic {
			good = (state == RX_STATE && waitforradio);
			if (good)
				waitforradio = FALSE;
		}

		if (good)
			do_next_rx();
		else //if (!(good && post do_next_rx_task()))
		{

#ifdef RADIO_DEBUG
			if (call DiagMsg.record()) {
				call DiagMsg.str("FFIT error! state="); 
				call DiagMsg.uint8(state); 
				call DiagMsg.uint8(substate); 
				call DiagMsg.uint8((uint8_t)(rxbyteptr-((uint8_t *) rxbufptr)));
				call DiagMsg.send();
			}
#endif
		}
		
		return SUCCESS;
	}

	async event result_t IA4420Interrupt.RGIT() {
		uint8_t good;
		atomic {
			good = (state == TX_STATE && waitforradio);
			if (good)
				waitforradio = FALSE;
		}
		
		if (good)
			do_next_tx();
		else   //if (!(good && post do_next_tx_task()))
		{
#ifdef RADIO_DEBUG
			if (call DiagMsg.record()) {
				call DiagMsg.str("RGIT error! state="); 
				call DiagMsg.uint8(state); 
				call DiagMsg.uint8(substate); 
				call DiagMsg.uint8((uint8_t)(rxbyteptr-((uint8_t *) rxbufptr)));
				call DiagMsg.send();
			}
#endif
		}
		return SUCCESS;
	}

	async event result_t IA4420Interrupt.POR() {
		return post inittask();
	}

	async event result_t IA4420Interrupt.RGUR() {
#ifdef RADIO_DEBUG
		if (call DiagMsg.record()) {
			call DiagMsg.str("RGUR! state="); 
			call DiagMsg.uint8(state); 
			call DiagMsg.uint8(substate); 
			call DiagMsg.uint8((uint8_t)(rxbyteptr-((uint8_t *) rxbufptr)));
			call DiagMsg.send();
		}
#endif
		return SUCCESS;
	}

	async event result_t IA4420Interrupt.FFOV() {
#ifdef RADIO_DEBUG
		if (call DiagMsg.record()) {
			call DiagMsg.str("FFOV! state="); 
			call DiagMsg.uint8(state); 
			call DiagMsg.uint8(substate); 
			call DiagMsg.uint8((uint8_t)(rxbyteptr-((uint8_t *) rxbufptr)));
			call DiagMsg.uint8(rxactive); 
			call DiagMsg.send();
		}
#endif
		substate = 0;
		waitforradio = FALSE;
		do_next_rx();
		
		return SUCCESS;
	}

	async event result_t IA4420Interrupt.WKUP() {
		return SUCCESS;
	}

	async event result_t IA4420Interrupt.EXT() {
		return SUCCESS;
	}

	async event result_t IA4420Interrupt.LBD() {
		return SUCCESS;
	}
/*
	event result_t Debug.outputComplete(result_t success) {
		return SUCCESS;
	}
*/


	/** Radio command has been sent:  **/
	async event result_t HPLIA4420Command.commandDone() {
		uint8_t lstate, lsubstate;
		
		//atomic {
			lstate = state;
			lsubstate = substate;
		//}

		if (lstate == INIT_STATE) {
			atomic sync_done = 1;
		}
		else if (lstate==RX_STATE) {
			switch(lsubstate)
			{
				case 0:
					substate = 1;
					do_next_rx(); //post do_next_rx_task();
					break;

				case 1:
					substate = 2;
					do_next_rx(); //post do_next_rx_task();
					break;

				case 2:
					substate = 3;
					waitforradio = TRUE;
					//post wait_for_radio_and_do_next_rx_task(); // wait for sync pattern
					break;
					
				case 6:
//					substate = 1;
//					do_next_rx();

//					rxactive = FALSE;
//					if (txstate==TX_PENDING)	// If transmission pending, begin wait&listen
//					begintxwait();

					post PacketRcvd();
					break;
			}
		}

		else if (lstate==TX_STATE) {

			switch(lsubstate)
			{
				case 0:
					substate = lsubstate+1;
					do_next_tx(); //post do_next_tx_task();
					substate=1;					
					break;
				case 1: // preamble1
				case 2: // preamble2
					substate = lsubstate+1;
					do_next_tx(); //post do_next_tx_task();
					break;

				case 3: // TX on
					substate = lsubstate+1;
					waitforradio = TRUE;
					break;

				case 4: // preamble3
					substate = lsubstate+1;
					waitforradio = TRUE;
					break;

				case 5:
					substate = 10;
					waitforradio = TRUE;
					break;					

				case 6:
				case 7:
				case 8:
				case 9:
		//			com_r04_s01__Write_Transmit_data (0xAA);
					substate = lsubstate+1;
					waitforradio = TRUE;
					break;
				
				/*** Sending syncron pattern ***/	
				case 10:
		//			com_r04_s01__Write_Transmit_data (0x2D);
					substate = lsubstate+1;
					waitforradio = TRUE;
					break;
				case 11:
		//			com_r04_s01__Write_Transmit_data (0xD4);
					substate = lsubstate+1;
					waitforradio = TRUE;
					break;

				case 12:
					waitforradio = TRUE;
					break;

				//CRC:
				case 50:
		//			com_r04_s01__Write_Transmit_data ((uint8_t)(pMsg->crc));
					substate = lsubstate+1;
					waitforradio = TRUE;

					break;
				case 51:
		//			com_r04_s01__Write_Transmit_data ((uint8_t)(pMsg->crc >> 8));
					substate = lsubstate+1;
					waitforradio = TRUE;
					break;
		
				case 52:
		//			com_r04_s01__Write_Transmit_data (0);
					substate = lsubstate+1;
					
					waitforradio = TRUE;
					break;

				case 53:
					post signalsenddone();
					atomic {
						txstate = TX_IDLE;
						state = RX_STATE;
						substate = 0;

//						state = prev_state;
//						substate=0;
//						if (state == RX_STATE)
//							post do_next_rx_task();
					}
					do_next_rx();

					break;
					
			}
		
		} 
		
		return SUCCESS;
	}
}
