#######################################################
#
# Family Tree generation program, v2.0
# Written by Ferenc Bodon and Simon Ward, March 2000 (simonward.com)
# Copyright (C) 2000 Ferenc Bodon, Simon K Ward
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# For a copy of the GNU General Public License, visit 
# http://www.gnu.org or write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#
#######################################################


package FamilyTreeGraphics;
@EXPORT = qw(new main);
use strict;
use warnings;
use FamilyTreeBase;
use Params::Validate qw(:all);
use List::Util qw(max);
my $q = new CGI;

#######################################################
#
# The HTML output table is generated in three parts:
#   - the Ancestor tree (ATree)
#   - the peer level (peers)
#   - the Descendant tree (DTree)
#
#
#######################################################
use base 'FamilyTreeBase';
sub new {
  my $type = shift;
  my $self = $type->SUPER::new(@_);
  $self->{personpageScript} = "person_page.cgi";
  $self->{target_person} = undef;
  $self->{peers}         = [];
  $self->{ATree}         = [];
  $self->{DTree}         = [];
  $self->{DLevels}       = 0;        # nr of levels in DTree
  $self->{DWidth}        = undef;    # width of DTree
  $self->{AWidth}        = undef;    # width of ATree
  $self->{cellwidth}     = undef;    # width of a cell
  $self->{gridWidth}     = undef;    # width of the tree
  $self->{fontsize}      = undef;
  
  $self->{NONE_person}->set_mother($self->{NONE_person});
  $self->{NONE_person}->set_father($self->{NONE_person});

  return $self;
}

sub main {
  my ($self) = validate_pos(@_, {type => HASHREF});
  $self->process_parameters();
  $self->target_check();
  $self->set_size();
  $self->password_check();
  @{ $self->{peers} } = $self->{family_tree_data}->get_peers( $self->{target_person} );
  if ( $self->{reqLevels} > 0 ) {
      $self->populate_tree_data_structure();
      $self->draw_family_tree( );
  }
  else {
  	  my $address = $q->url(-relative=>0);
  	  $address =~ s/$self->{treeScript}/$self->{personpageScript}/;
  	  $address .= "?target=".$self->{target_person}->get_id()
  	     .";lang=".$self->{lang};
  	  print $q->redirect($address);
  }
  $self->endpage();
}

#######################################################
# processing the parameters (type and passwd)
sub process_parameters {
  my ($self) = validate_pos(@_, {type => HASHREF});
  $self->SUPER::process_parameters();
  my $id = CGI::param("target");
  $self->{target_person} = $self->{family_tree_data}{people}{$id};
  $self->{reqLevels}     = CGI::param("levels");
  $self->{reqLevels}     = 2 unless ( defined $self->{reqLevels} );
}

#######################################################
# check if target person exists in database
sub target_check {
  my ($self) = validate_pos(@_, {type => HASHREF});
  if ( !defined $self->{target_person} ) {
    my $title = $self->{textGenerator}->noDataAbout( CGI::param("target") );
    $self->toppage($title);
    print $q->br, $title, $q->br, "\n";
    $self->endpage();
    exit 1;
  }
}

#######################################################
# Size the output according to the no. levels being displayed
sub set_size {
  my ($self) = validate_pos(@_, {type => HASHREF});
  if ( $self->{reqLevels} > 3 ) {
    $self->{imgwidth}  = 45;
    $self->{fontsize}  = 1;
  }
  elsif ( $self->{reqLevels} == 3 ) {
    $self->{imgwidth}  = 60;
    $self->{fontsize}  = 2;
  }
  elsif ( $self->{reqLevels} == 2 ) {
    $self->{imgwidth}  = 90;
    $self->{fontsize}  = 3;
  }
  elsif ( $self->{reqLevels} == 1 ) {
    $self->{imgwidth}  = 110;
    $self->{fontsize}  = 2;
  }
  elsif ( $self->{reqLevels} == 0 ) {
    $self->{imgwidth}  = 240;
    $self->{fontsize}  = 2;
  }
  else {
    $self->{cellwidth} = 70;
    $self->{imgwidth}  = 60;
    $self->{fontsize}  = 2;
  }
  $self->{cellwidth} = "100%";
  $self->{imgheight} = $self->{imgwidth} * 1.5;
}

#######################################################
# populate tree data structures
sub populate_tree_data_structure {
  my ( $self) = validate_pos(@_, {type => HASHREF});

  $self->{gridWidth} =
    $self->getGridWidth( $self->{target_person}, $self->{reqLevels} );

  $self->fillATree( $self->{target_person}, 0, $self->{reqLevels} );
  $self->fillDTree( $self->{target_person}, 0, $self->{reqLevels} );
}

sub html_img {
  my ( $self, $person ) = validate_pos(@_, 
      {type => HASHREF}, {type => SCALARREF});
  my $img = $self->SUPER::html_img($person);
  return ($person == $self->{target_person} || 
      $person == $self->{NONE_person}) ? $img : $self->aref_tree($img, $person);   
}

sub img_graph {
  my ( $self, $graphics ) = validate_pos(@_,
    {type => HASHREF}, {type => SCALAR}, 0);
  return $q->img(
          {
            -width => $self->{cellwidth},
            -height=> "26",
            -src   => "$self->{graphicsUrl}/".$graphics.".gif",
            -alt   => "",
          } );
}
sub hone_img_graph {
  my ( $self ) = validate_pos(@_, {type => HASHREF}, 0);
  return $self->img_graph("hone");
}
sub getATreeWidth {
  my ( $self, $levels ) = validate_pos(@_, 
    {type => HASHREF}, {type => SCALAR}, 0);
  return 2**( $levels );
}
#######################################################
# returns the width of tree below this person
# root_person:  this person
# levels:     no. of levels to descend in tree
sub getDTreeWidth {
  my ( $self, $levels, $root_person ) = validate_pos(@_, 
    {type => HASHREF}, {type => SCALAR}, {type => SCALARREF} );

#print "called: getDTreeWidth with \$root_node = $root_person->get_full_name(), \$levels = $levels\n";

  if ( 0 == $levels || 
      $root_person == $self->{NONE_person} || 
      0 == $self->{family_tree_data}->numChildren($root_person) )
  {
    # No further levels to show
    return 1;
  }
  else {
    my $width = 0;
    $width += $self->getDTreeWidth( $levels - 1, $_ )
      for ( @{ $self->{family_tree_data}{children}{ $root_person->get_id() } } );
    return $width;
  }
}

#######################################################
# returns the no. levels available in Ancestor tree
#   above this person
# root_person:  this person
# anc_level:  current level of ancestor tree (0=root_node)
# req_levels: no. levels requested
sub getATreeLevels {
  my ( $self, $root_person, $anc_level, $req_levels ) = validate_pos(@_, 
    {type => HASHREF}, {type => SCALARREF|UNDEF}, {type => SCALAR}, {type => SCALAR} );

#  print "called: getATreeLevels (root_node=$root_person->get_name()->get_full_name(), anc_level=$anc_level,  req_levels=$req_levels)\n";
  if ( $req_levels == 0 ) {
    return 0;
  }
  elsif ( !defined $root_person || !defined $root_person->get_father() && 
    !defined $root_person->get_mother() || $anc_level == $req_levels ) {
    return $anc_level;
  }
  else {
    my $p1_levels = $self->getATreeLevels( $root_person->get_father(),
      $anc_level + 1, $req_levels );
    my $p2_levels = $self->getATreeLevels( $root_person->get_mother(),
      $anc_level + 1, $req_levels );
    return List::Util::max ($p1_levels, $p2_levels);
  }
}

#######################################################
# populate the Descendant Tree structure for all
#   people below the person specified
# $root_person:  this person
# dec_level:  current level of descendant tree (0=root_node)
# req_levels: no. levels requested
sub fillDTree {
  my ( $self, $root_person, $dec_level, $req_levels ) = validate_pos(@_, 
    {type => HASHREF}, {type => SCALARREF}, {type => SCALAR}, {type => SCALAR} );

#  print "called: fillDTree (root_node=$root_node_id, dec_level=$dec_level,  req_levels=$req_levels)\n";
  $dec_level++;
  my $level_ok = 0;

  if ( $root_person != $self->{NONE_person}
    && $self->{family_tree_data}->numChildren($root_person) > 0  )
  {
    push @{ $self->{DTree}[$dec_level] },
      @{ $self->{family_tree_data}{children}{ $root_person->get_id() } };
    $level_ok = 1;
  }
  else {
    push @{ $self->{DTree}[$dec_level] }, $self->{NONE_person};
    $self->{family_tree_data}{children}{ $root_person->get_id() } =
      [ $self->{NONE_person} ];
  }
  if ( $dec_level < $req_levels ) {
    $self->fillDTree( $_, $dec_level, $req_levels )
      for ( @{ $self->{family_tree_data}{children}{ $root_person->get_id() } } );
  }
  $self->{DLevels} = $dec_level
    if ( $dec_level > $self->{DLevels} && $level_ok == 1 );
}

sub putNTD {
  my ( $self, $n, $data ) = validate_pos(@_, 
    {type => HASHREF}, {type => SCALAR}, {type => SCALAR, default => ""} );
    print $q->td($data), "\n" for (1 .. $n);
}
sub drawRow {
  my ( $self, $used_width, $people, $diff_levels, $this_level,
    $left_fill, $emptyTDCond, $group_width_func, $display_func ) = validate_pos(@_, 
    {type => HASHREF}, {type => SCALAR}, {type => ARRAYREF}, {type => SCALAR}, 
    {type => SCALAR}, {type => SCALAR}, {type => CODEREF}, {type => CODEREF}, {type => CODEREF} );
  my $right_fill = $self->{gridWidth} - $used_width - $left_fill;
  my $blank_line = 1;

  print $q->start_Tr, "\n";
  $self->putNTD($left_fill);
  foreach my $person (@$people) {
    my $group_width  = $group_width_func->($self, $diff_levels, $person);
    my $left  = int( ( $group_width - 1 ) / 2 );
    my $right = $group_width - 1 - $left;
    
    $self->putNTD($left);
    if ( $emptyTDCond->($self, $person, $this_level) ) {
      print $q->td(), "\n";
    }
    else {
      print $q->td( {-align => "center" },
        $display_func->($self, $person) );
      $blank_line = 0;
    }
    $self->putNTD($right);
  }
  $self->putNTD($right_fill);
  print $q->end_Tr, "\n";
  return $blank_line;
}
sub unknownEquiCond {
  my ( $self, $person ) = validate_pos(@_, 
    {type => HASHREF}, {type => SCALARREF}, 0 );
  return $person == $self->{NONE_person};
}
sub unknownEquiNoChildrenCond {
  my ( $self, $person, $this_level ) = validate_pos(@_, 
    {type => HASHREF}, {type => SCALARREF}, SCALAR );
  return $person == $self->{NONE_person} || 
    ( $self->{family_tree_data}->numChildren($person) > 0 
      && $self->{family_tree_data}{children}{ $person->get_id() }[0] == $self->{NONE_person} ) || 
    ( $this_level == $self->{reqLevels} );
}
sub falseCond {
  my ( $self ) = @_;
  return 0;
}
#######################################################
# generate a line of the D-tree graphics OVER the
# level specified
# this_level: level of grid to generate
# max_levels: max depth that will be shown
sub getDGridLineG {
  my ( $self, $this_level, $max_levels ) = validate_pos(@_, 
    {type => HASHREF}, SCALAR, SCALAR );

#  print "called: getDGridLineG (this_level = $this_level, max_levels = $max_levels)\n";
  my ( $left_fill, $branch, $right_fill );
  my $lefto_fill  = int( ( $self->{gridWidth} - $self->{DWidth} ) / 2 );
  my $righto_fill = $self->{gridWidth} - $self->{DWidth} - $lefto_fill;

  # Spacers on LHS - fills gap between overall grid width and width of Dgrid
  print $q->start_Tr, "\n";
  $self->putNTD($lefto_fill);

  if ( @{ $self->{DTree}[$this_level] } == 0 ) {
    printf "|;";
  }
  else {
    foreach my $person (@{ $self->{DTree}[$this_level] }) {
      # Find which parent is in the level above...
      my $this_p1 = $person->get_father();
      my $this_p2 = $person->get_mother();
      my $this_parent;
      #printf "looking for $this_p1 or $this_p2 in row above...\n";
      if ( 1 == $this_level ) {
        $this_parent = $self->{target_person};
      }
      else {
        foreach my $person2 (@{ $self->{DTree}[$this_level - 1] }) {
          if ( ( defined $this_p1 ) && ( $person2 == $this_p1 ) )
          {
            $this_parent = $this_p1;
            last;
          }
          elsif ( ( defined $this_p2 ) && ( $person2 == $this_p2 ) )
          {
            $this_parent = $this_p2;
            last;
          }
        }
      }

      # Is this person the first child of this parent?
      if ( $person != $self->{NONE_person} ) {
        if ( $person == $self->{family_tree_data}{children}{ $this_parent->get_id() }[0] )
        {
          $left_fill = "";
          $branch    = $self->img_graph("hleft");
          $right_fill = $self->img_graph("hblank");
        }

        # Is this person the last child of this parent?
        elsif ( $person == $self->{family_tree_data}{children}{ $this_parent->get_id() }[-1] )
        {
          $left_fill = $self->img_graph("hblank"); 
          $branch = $self->img_graph("hright");
          $right_fill = "";
        }
        else {
          $left_fill = $right_fill = $self->img_graph("hblank");
          $branch = $self->img_graph("hbranch");
        }
      }
      if ( $person == $self->{NONE_person} ) {
        # This blank person
        $left_fill  = $branch = $right_fill = "";
      }
      elsif ( 1 == @{ $self->{family_tree_data}{children}{ $this_parent->get_id() } } )
      {
        # This person is an only child
        $left_fill = $right_fill = "";
        $branch    = $self->img_graph("hone");
      }
      my $group_width = $self->getDTreeWidth( $max_levels - $this_level, $person );
      my $left  = int( ( $group_width - 1 ) / 2 );
      my $right = $group_width - 1 - $left;

      $self->putNTD( $left, $left_fill );
      print $q->td($branch);
      $self->putNTD( $right, $right_fill );
    }
  }

  # Spacers on RHS - fills gap between overall grid width and width of Dgrid
  $self->putNTD($righto_fill);
  print $q->end_Tr, "\n";
}

#######################################################
# build A-tree for this person
# root_node: this person
# anc_level:  current level of ancestor tree (0=root node)
# req_levels: no. levels requested
sub fillATree {
  my ( $self, $root_person, $anc_level, $req_levels ) = validate_pos(@_, 
    {type => HASHREF}, {type => SCALARREF|UNDEF}, SCALAR, SCALAR );

#  print "called: fillATree (root_node = $root_person, anc_level = $anc_level, req_levels = $req_levels)\n";
  $anc_level++;
  if(defined $root_person) {
    push @{ $self->{ATree}[$anc_level] }, defined $root_person->get_father() ?
      $root_person->get_father() : $self->{NONE_person};
    push @{ $self->{ATree}[$anc_level] }, defined $root_person->get_mother() ?
      $root_person->get_mother() : $self->{NONE_person};
    if ( $anc_level < $req_levels ) {
      $self->fillATree( $root_person->get_father(), $anc_level, $req_levels );
      $self->fillATree( $root_person->get_mother(), $anc_level, $req_levels );
    }    
  }
  else {
    push @{ $self->{ATree}[$anc_level] }, 
      ($self->{NONE_person}, $self->{NONE_person});
      if ( $anc_level < $req_levels ) {
        $self->fillATree( $self->{NONE_person}, $anc_level, $req_levels );
      $self->fillATree( $self->{NONE_person}, $anc_level, $req_levels );
      }  
  }
}

#######################################################
# get the overall width of the grid
sub getGridWidth {
  my ( $self, $root_node, $levels ) = validate_pos(@_, 
    {type => HASHREF}, {type => SCALARREF}, SCALAR);

#  print "called getDTreeWidth with \$root_node= $root_node, \$levels= $levels";

  $self->{DWidth} = $self->getDTreeWidth( $levels, $root_node  );
  $self->{AWidth} = 2**$self->getATreeLevels( $root_node, 0, $levels );
  my $PWidth = $self->getPTreeWidth();

  return List::Util::max ($self->{DWidth}, $self->{AWidth}, $PWidth);
}

#######################################################
# draw the graphics UNDER the level specified
# this_level: level of grid to generate
# max_levels: max depth that will be shown
sub getAGridLineG {
  my ( $self, $this_level, $max_levels ) = validate_pos(@_, 
    {type => HASHREF}, SCALAR, SCALAR);

  if ( $this_level > $max_levels ) {
    return;
  }

  my $left_fill  = int( ( $self->{gridWidth} - $self->{AWidth} + 1 ) / 2 );
  my $right_fill = $self->{gridWidth} - $self->{AWidth} - $left_fill;

  print $q->start_Tr, "\n";
  $self->putNTD($left_fill);

  for ( my $index = 0; $index < @{ $self->{ATree}[$this_level] }; $index += 2 )
  {
    my $node_width = 2**( $max_levels - $this_level );
    my $nodel_fill = int( ( $node_width - 1 ) / 2 );
    my $noder_fill = $node_width - 1 - $nodel_fill;

    $self->putNTD($nodel_fill);
    print $q->td( $self->img_graph("hleftup")),"\n";
    $self->putNTD( $noder_fill + $nodel_fill, $self->img_graph("hblankup") );  
    print $q->td( $self->img_graph("hrightup") ), "\n";
    $self->putNTD($noder_fill);
  }

  $self->putNTD($right_fill);
  print $q->end_Tr, "\n";

  print $q->start_Tr, "\n";
  $self->putNTD($left_fill);

  for ( my $index = 0 ; $index <= $#{ $self->{ATree}[$this_level] }; $index += 2 )
  {
    my $node_width = 2**( $max_levels - $this_level );
    my $nodel_fill = int( ( $node_width - 1 ) / 2 );
    my $noder_fill = $node_width - 1 - $nodel_fill;

    $self->putNTD( $node_width - 1 );
    print $q->td( $self->img_graph("hone") ), "\n";
    $self->putNTD($node_width);
  }
  $self->putNTD($right_fill);
  print $q->end_Tr, "\n";
}

#######################################################
sub buildDGrid {
  my ($self) = validate_pos(@_, {type => HASHREF});

  for ( my $this_level = 1 ; $this_level <= $self->{DLevels} ; ++$this_level ) {
    $self->getDGridLineG( $this_level, $self->{reqLevels} );
    my $left_fill = int( ( $self->{gridWidth} - $self->{DWidth} ) / 2 );
    my $blank_line = $self->drawRow($self->{DWidth}, \@{ $self->{DTree}[$this_level] }, 
      $self->{reqLevels} - $this_level, $this_level, $left_fill, 
      \&unknownEquiCond, \&getDTreeWidth, \&FamilyTreeGraphics::html_img);
    $self->drawRow($self->{DWidth}, \@{ $self->{DTree}[$this_level] },
      $self->{reqLevels} - $this_level, $this_level, $left_fill, 
      \&unknownEquiCond, \&getDTreeWidth, \&html_name);
    $self->drawRow($self->{DWidth}, \@{ $self->{DTree}[$this_level] },
      $self->{reqLevels} - $this_level, $this_level, $left_fill, 
      \&unknownEquiNoChildrenCond, \&getDTreeWidth, \&hone_img_graph);
    $self->{DLevels} = $this_level - 1 if ( $blank_line == 1 );   
  }
}

#######################################################
sub buildAGrid {
  my ( $self ) = validate_pos(@_, {type => HASHREF});
#printf "calling: getAGridLine \n";

  my $ATreeLevels =
    $self->getATreeLevels( $self->{target_person}, 0, $self->{reqLevels} );
  for ( my $this_level = $ATreeLevels ; $this_level > 0 ; --$this_level ) {
    my $used_width = 2 ** $ATreeLevels;
    my $left_fill  = int( ( $self->{gridWidth} - $used_width + 1 ) / 2 );
    $self->drawRow($used_width, \@{ $self->{ATree}[$this_level] },
      $ATreeLevels - $this_level, $this_level, $left_fill, 
      \&falseCond, \&getATreeWidth , \&FamilyTreeGraphics::html_img);
    $self->drawRow($used_width, \@{ $self->{ATree}[$this_level] },
      $ATreeLevels - $this_level, $this_level, $left_fill, 
      \&falseCond, \&getATreeWidth, \&html_name);
    $self->getAGridLineG( $this_level, $ATreeLevels );
  }

  #printf "buildAGrid returns";
}

#######################################################
sub buildPGrid {
  my ($self) = validate_pos(@_, {type => HASHREF});
  my $node_pos = 0;

  my $PWidth     = $self->getPTreeWidth();
  my $left_fill  = int(( $self->{gridWidth} - $PWidth ) / 2 );
  my $right_fill = $self->{gridWidth} - $PWidth - $left_fill;

  ++$node_pos while($self->{peers}[$node_pos] != $self->{target_person});
    
  my $left_side   = $node_pos;
  my $right_side  = $#{ $self->{peers} } - $node_pos;
  my $leftn_fill  = int( ( ( $PWidth - 1 ) / 2 ) - $left_side );
  my $rightn_fill = int( ( ($PWidth) / 2 ) - $right_side );

  print $q->start_Tr, "\n";
  $self->putNTD($left_fill + $leftn_fill);

  if ( @{ $self->{peers} } > 1 ) {
    print $q->td( $self->img_graph("hleft") ), "\n";
    $self->putNTD($#{ $self->{peers} }-1, $self->img_graph("hbranch"));
    print $q->td( $self->img_graph("hright") ),  "\n";    
  }
  else {
    print $q->td( $self->img_graph("hone") ), "\n";
  }
  $self->putNTD($rightn_fill + $right_fill);
  print $q->end_Tr, "\n";

  print $q->start_Tr, "\n";
  $self->putNTD($left_fill + $leftn_fill);
  print $q->td( {-align => "center" }, $self->html_img( $_ ) )
    for(@{ $self->{peers} });
  $self->putNTD($rightn_fill + $right_fill);
  print $q->end_Tr, "\n";

  print $q->start_Tr, "\n";
  $self->putNTD($left_fill + $leftn_fill);

  print $q->td( { -align => "center" }, $self->html_name( $_ ) ) 
    for(@{ $self->{peers} });
  $self->putNTD($rightn_fill + $right_fill);
  print $q->end_Tr, "\n";

  if ( $self->{family_tree_data}{children}{ $self->{target_person}->get_id() }[0] !=
    $self->{NONE_person} )
  {
    print $q->start_Tr, "\n";
    my $gridLeft = int( ( $self->{gridWidth} - 1 ) / 2 );
    my $gridRight = $self->{gridWidth} - 1 - $gridLeft;
    $self->putNTD($gridLeft);
    print $q->td( $self->img_graph("hone") ), "\n";
    $self->putNTD($gridRight);
    print $q->end_Tr, "\n";
  }

}

#######################################################
# find the width of the peer line
# (allowing for the fact that it may be off-centre)
sub getPTreeWidth {
  my ($self) = validate_pos(@_, {type => HASHREF});
  my $node_pos = 0;
  ++$node_pos while($self->{peers}[$node_pos] != $self->{target_person});
  my $right_side = $#{ $self->{peers} } - $node_pos;
  my $big_side = List::Util::max ($node_pos, $right_side );  
  return ( $big_side * 2 ) + 1;
}

#######################################################
# generates the html for the name of this person
sub html_name {
  my ( $self, $person ) = validate_pos(@_, {type => HASHREF}, {type => SCALARREF});
  return $q->font({-size => $self->{fontsize}}, $self->{textGenerator}{Unknown})
    if ( !defined $person || $person == $self->{NONE_person} );
  my $show_name;
  if(defined $person->get_name()) {
    $show_name = ( $self->{reqLevels} > 1 ) ? 
      $person->get_name()->get_first_name() : $person->get_name()->get_short_name(); 
  } else {
    $show_name = $self->{textGenerator}{Unknown};
  } 
  if ( $person == $self->{target_person} ) {
    return $q->strong($q->font({-size => $self->{fontsize}}, $show_name));
  }
  else {
    return $q->font({-size => $self->{fontsize}}, $self->aref_tree($show_name, $person));    
  }
}


sub print_zoom_buttons {
  my ($self)     = validate_pos(@_, {type => HASHREF});
  my $lev_minus1 = $self->{reqLevels} - 1;
  my $lev_plus1  = $self->{reqLevels} + 1;

  print $q->start_table(
    { -border => "0", -cellpadding => "0", -cellspacing => "2" } ), "\n",
    $q->start_Tr;
  if ( $lev_minus1 >= 0 ) {
    print $q->start_td({-align => "center"}), "\n",
      $self->aref_tree($q->img( {
          -src => "$self->{graphicsUrl}/zoomin.gif",
          -alt => $self->{textGenerator}->ZoomIn($lev_minus1) }), $self->{target_person}, $lev_minus1),
      $q->end_td, "\n";
  }    
  print $q->start_td({-align => "center"}), "\n",
    $self->aref_tree($q->img( {
          -src => "$self->{graphicsUrl}/zoomout.gif",
          -alt => $self->{textGenerator}->ZoomOut($lev_plus1) }), $self->{target_person}, $lev_plus1),
      $q->end_td, $q->end_Tr, "\n",
      $q->end_table, $q->br, $q->br, "\n";
}
#########################################################
# OUTPUT SECTION                                        #
#########################################################
sub draw_family_tree {
  my ( $self ) = validate_pos(@_, {type => HASHREF});

  # header html for page
  my $title = $self->{textGenerator}->familyTreeFor( 
    $self->{target_person}->get_name()->get_full_name() );
  $self->toppage($title);

  # Zoom buttons
  print $q->start_center, "\n";
  $self->print_zoom_buttons();
  # Draw the grid
  print $q->start_table(
    { -border => "0", -cellpadding => "0", -cellspacing => "0" } ), "\n";
  
  $self->buildAGrid();
  $self->buildPGrid();
  $self->buildDGrid();
  print $q->end_table,  "\n",
    $q->end_center, "\n",
}

1;

